import { useState, useRef, useEffect } from "react";
import { Send, Sparkles, ExternalLink } from "lucide-react";
import { Button } from "./ui/button";
import { Textarea } from "./ui/textarea";
import { Badge } from "./ui/badge";
import { Comment, CommentChannelType } from "../types";
import { toast } from "sonner";

interface CommentViewProps {
  comment: Comment | null;
  onSendReply: (content: string) => void;
  onAIResponse: (content: string) => void;
}

export function CommentView({
  comment,
  onSendReply,
  onAIResponse,
}: CommentViewProps) {
  const [inputValue, setInputValue] = useState("");
  const [isGenerating, setIsGenerating] = useState(false);
  const repliesEndRef = useRef<HTMLDivElement>(null);
  const textareaRef = useRef<HTMLTextAreaElement>(null);

  const scrollToBottom = () => {
    repliesEndRef.current?.scrollIntoView({ behavior: "smooth" });
  };

  useEffect(() => {
    scrollToBottom();
  }, [comment?.replies]);

  const handleSend = () => {
    if (inputValue.trim() && comment) {
      onSendReply(inputValue);
      setInputValue("");
      toast.success("Yorum yanıtı gönderildi");
    }
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === "Enter" && !e.shiftKey) {
      e.preventDefault();
      handleSend();
    }
  };

  const generateAIResponse = async () => {
    if (!comment) return;

    setIsGenerating(true);
    toast.info("AI cevap hazırlıyor...");
    
    // Simulate AI thinking time
    await new Promise((resolve) => setTimeout(resolve, 1500));

    const commentText = comment.comment.toLowerCase();

    // Generate contextual AI responses for comments
    const responses = {
      fiyat: "Teşekkür ederiz! Ürün fiyatı ve detayları için size DM gönderdik. İnceleyip dönüş yapabilirsiniz. 😊",
      beden: "Merhaba! Bu ürün S, M, L ve XL bedenlerde mevcuttur. Detaylı beden tablosu için profilimize bakabilirsiniz. 👗",
      kargo: "Tüm siparişlerde kargo ücretsizdir! 🚚 Sipariş vermek için linke tıklayabilirsiniz.",
      kampanya: "Bu kampanya 15 gün daha devam edecek. Kaçırmayın! 🎉 Detaylar için size mesaj gönderdik.",
      stok: "Ürün stoklarda mevcut! Hemen sipariş verebilirsiniz. 📦 Link profilimizde.",
      positive: "Teşekkür ederiz! ❤️ Sizin gibi müşterilerimiz için en iyisini sunmak için çalışıyoruz. 🙏",
      default: "Teşekkür ederiz! İlginiz için size mesaj gönderdik. Detaylı bilgi almak için DM'den yazabilirsiniz. 😊",
    };

    let aiResponse = responses.default;

    if (commentText.includes("fiyat") || commentText.includes("kaç")) {
      aiResponse = responses.fiyat;
    } else if (commentText.includes("beden") || commentText.includes("numara")) {
      aiResponse = responses.beden;
    } else if (commentText.includes("kargo")) {
      aiResponse = responses.kargo;
    } else if (commentText.includes("kampanya") || commentText.includes("indirim") || commentText.includes("bitiyor")) {
      aiResponse = responses.kampanya;
    } else if (commentText.includes("stok") || commentText.includes("var mı")) {
      aiResponse = responses.stok;
    } else if (commentText.includes("😍") || commentText.includes("beğen") || commentText.includes("harika") || commentText.includes("muhteşem")) {
      aiResponse = responses.positive;
    }

    setInputValue(aiResponse);
    setIsGenerating(false);
    toast.success("AI cevabı hazır! İnceleyip gönderin.");
  };

  const getChannelColor = (channelType: CommentChannelType) => {
    const colors = {
      'instagram-comment': '#E4405F',
      'facebook-comment': '#1877F2',
    };
    return colors[channelType];
  };

  if (!comment) {
    return (
      <div className="flex-1 flex items-center justify-center bg-gray-50">
        <div className="text-center">
          <div className="w-16 h-16 bg-gray-200 rounded-full flex items-center justify-center mx-auto mb-4">
            <Send className="h-8 w-8 text-gray-400" />
          </div>
          <h3 className="text-lg font-medium text-gray-900 mb-2">
            Yorum Seçin
          </h3>
          <p className="text-sm text-gray-500">
            Sol taraftan bir yorum seçerek başlayın
          </p>
        </div>
      </div>
    );
  }

  return (
    <div className="flex-1 flex flex-col bg-white">
      {/* Header */}
      <div className="p-4 border-b border-gray-200 bg-gray-50">
        <div className="flex items-center gap-3 mb-3">
          <div className="w-10 h-10 rounded-full bg-gray-200 flex items-center justify-center">
            <span className="text-sm font-semibold text-gray-600">
              {comment.customerName.charAt(0).toUpperCase()}
            </span>
          </div>
          <div>
            <h3 className="font-semibold text-gray-900">@{comment.customerName}</h3>
            <div className="flex items-center gap-2">
              <Badge
                style={{
                  backgroundColor: `${getChannelColor(comment.channelType)}20`,
                  color: getChannelColor(comment.channelType),
                }}
              >
                {comment.channelName}
              </Badge>
            </div>
          </div>
        </div>

        {/* Post Info */}
        <div className="bg-white p-3 rounded-lg border border-gray-200">
          <div className="flex items-start justify-between">
            <div className="flex-1">
              <p className="text-xs text-gray-500 mb-1">Gönderi</p>
              <p className="text-sm font-medium text-gray-900">{comment.postTitle}</p>
            </div>
            <Button variant="ghost" size="sm">
              <ExternalLink className="h-4 w-4" />
            </Button>
          </div>
        </div>
      </div>

      {/* Original Comment & Replies */}
      <div className="flex-1 overflow-y-auto p-4 bg-gray-50">
        <div className="space-y-4 max-w-4xl mx-auto">
          {/* Original Comment */}
          <div className="bg-white p-4 rounded-lg border border-gray-200 shadow-sm">
            <div className="flex items-start gap-2 mb-2">
              <div className="w-8 h-8 rounded-full bg-gray-200 flex items-center justify-center">
                <span className="text-xs font-semibold text-gray-600">
                  {comment.customerName.charAt(0).toUpperCase()}
                </span>
              </div>
              <div className="flex-1">
                <p className="text-sm font-medium text-gray-900">@{comment.customerName}</p>
                <p className="text-xs text-gray-500">
                  {comment.timestamp.toLocaleString("tr-TR", {
                    day: "numeric",
                    month: "short",
                    hour: "2-digit",
                    minute: "2-digit",
                  })}
                </p>
              </div>
            </div>
            <p className="text-sm text-gray-900">{comment.comment}</p>
          </div>

          {/* Replies */}
          {comment.replies.map((reply) => (
            <div
              key={reply.id}
              className={`flex ${reply.sender === "agent" ? "justify-end" : "justify-start"}`}
            >
              <div
                className={`max-w-[70%] rounded-lg p-3 shadow-sm ${
                  reply.sender === "agent"
                    ? "bg-blue-600 text-white"
                    : "bg-white text-gray-900 border border-gray-200"
                }`}
              >
                <p className="text-sm">{reply.content}</p>
                <p
                  className={`text-xs mt-1 ${
                    reply.sender === "agent" ? "text-blue-100" : "text-gray-500"
                  }`}
                >
                  {reply.timestamp.toLocaleTimeString("tr-TR", {
                    hour: "2-digit",
                    minute: "2-digit",
                  })}
                </p>
              </div>
            </div>
          ))}
          <div ref={repliesEndRef} />
        </div>
      </div>

      {/* Input Area */}
      <div className="p-4 border-t border-gray-200 bg-white">
        <div className="max-w-4xl mx-auto">
          <div className="flex gap-2 mb-3">
            <Button
              variant="outline"
              size="sm"
              onClick={generateAIResponse}
              disabled={isGenerating}
              className="flex items-center gap-2"
            >
              <Sparkles className="h-4 w-4" />
              {isGenerating ? "AI Hazırlıyor..." : "AI ile Cevap Oluştur"}
            </Button>
          </div>
          <div className="flex gap-2">
            <Textarea
              ref={textareaRef}
              placeholder="Yoruma yanıt yazın... (Enter ile gönderin)"
              value={inputValue}
              onChange={(e) => setInputValue(e.target.value)}
              onKeyPress={handleKeyPress}
              className="resize-none"
              rows={3}
            />
            <Button
              onClick={handleSend}
              disabled={!inputValue.trim()}
              className="flex-shrink-0 h-auto"
            >
              <Send className="h-4 w-4" />
            </Button>
          </div>
        </div>
      </div>
    </div>
  );
}