import { useState, useEffect } from "react";
import { useParams } from "react-router";
import { CommentList } from "./CommentList";
import { CommentView } from "./CommentView";
import { mockComments } from "../data/mockData";
import { Comment } from "../types";
import { Filter, Calendar } from "lucide-react";
import { Button } from "./ui/button";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "./ui/select";
import { Input } from "./ui/input";
import { Popover, PopoverContent, PopoverTrigger } from "./ui/popover";

export function Comments() {
  const { channelId } = useParams();
  const [selectedComment, setSelectedComment] = useState<Comment | null>(null);
  const [comments, setComments] = useState(mockComments);
  const [channelFilter, setChannelFilter] = useState<string>("all");
  const [startDate, setStartDate] = useState<string>("");
  const [endDate, setEndDate] = useState<string>("");

  // Filter comments by channel if channelId is provided
  let filteredComments = channelId
    ? comments.filter((comment) => comment.channelType === channelId)
    : comments;

  // Apply channel filter
  if (channelFilter !== "all") {
    filteredComments = filteredComments.filter(
      (comment) => comment.channelType === channelFilter
    );
  }

  // Apply date range filter
  if (startDate) {
    const startDateTime = new Date(startDate).getTime();
    filteredComments = filteredComments.filter(
      (comment) => comment.timestamp.getTime() >= startDateTime
    );
  }
  if (endDate) {
    const endDateTime = new Date(endDate).setHours(23, 59, 59, 999);
    filteredComments = filteredComments.filter(
      (comment) => comment.timestamp.getTime() <= endDateTime
    );
  }

  useEffect(() => {
    if (filteredComments.length > 0 && !selectedComment) {
      setSelectedComment(filteredComments[0]);
    }
  }, [channelId]);

  const handleSelectComment = (comment: Comment) => {
    setSelectedComment(comment);
  };

  const handleSendReply = (content: string) => {
    if (!selectedComment) return;

    const newReply = {
      id: `reply-${Date.now()}`,
      sender: 'agent' as const,
      content,
      timestamp: new Date(),
    };

    setComments((prevComments) =>
      prevComments.map((comment) =>
        comment.id === selectedComment.id
          ? {
              ...comment,
              replies: [...comment.replies, newReply],
              unread: 0,
            }
          : comment
      )
    );

    setSelectedComment((prev) =>
      prev
        ? {
            ...prev,
            replies: [...prev.replies, newReply],
          }
        : null
    );
  };

  const handleAIResponse = (aiContent: string) => {
    if (!selectedComment) return;

    const aiReply = {
      id: `ai-${Date.now()}`,
      sender: 'agent' as const,
      content: aiContent,
      timestamp: new Date(),
    };

    setComments((prevComments) =>
      prevComments.map((comment) =>
        comment.id === selectedComment.id
          ? {
              ...comment,
              replies: [...comment.replies, aiReply],
              unread: 0,
            }
          : comment
      )
    );

    setSelectedComment((prev) =>
      prev
        ? {
            ...prev,
            replies: [...prev.replies, aiReply],
          }
        : null
    );
  };

  return (
    <div className="h-full flex flex-col">
      {/* Filter Bar */}
      <div className="bg-white border-b border-gray-200 p-4">
        <div className="flex items-center gap-4 flex-wrap">
          <div className="flex items-center gap-2">
            <Filter className="h-4 w-4 text-gray-500" />
            <span className="text-sm font-medium text-gray-700">Filtrele:</span>
          </div>

          <Select value={channelFilter} onValueChange={setChannelFilter}>
            <SelectTrigger className="w-48">
              <SelectValue placeholder="Kanal" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">Tüm Kanallar</SelectItem>
              <SelectItem value="instagram-comment">📸 Instagram</SelectItem>
              <SelectItem value="facebook-comment">👥 Facebook</SelectItem>
            </SelectContent>
          </Select>

          <Popover>
            <PopoverTrigger asChild>
              <Button variant="outline" size="sm" className="gap-2">
                <Calendar className="h-4 w-4" />
                Tarih Aralığı
              </Button>
            </PopoverTrigger>
            <PopoverContent className="w-80">
              <div className="space-y-4">
                <div>
                  <label className="text-sm font-medium text-gray-700 mb-2 block">
                    Başlangıç Tarihi
                  </label>
                  <Input
                    type="date"
                    value={startDate}
                    onChange={(e) => setStartDate(e.target.value)}
                  />
                </div>
                <div>
                  <label className="text-sm font-medium text-gray-700 mb-2 block">
                    Bitiş Tarihi
                  </label>
                  <Input
                    type="date"
                    value={endDate}
                    onChange={(e) => setEndDate(e.target.value)}
                  />
                </div>
                <Button
                  variant="outline"
                  size="sm"
                  className="w-full"
                  onClick={() => {
                    setStartDate("");
                    setEndDate("");
                  }}
                >
                  Tarihleri Temizle
                </Button>
              </div>
            </PopoverContent>
          </Popover>

          {(channelFilter !== "all" || startDate || endDate) && (
            <Button
              variant="ghost"
              size="sm"
              onClick={() => {
                setChannelFilter("all");
                setStartDate("");
                setEndDate("");
              }}
              className="text-blue-600 hover:text-blue-700"
            >
              Filtreleri Temizle
            </Button>
          )}
        </div>
      </div>

      <div className="flex-1 flex overflow-hidden">
        <CommentList
          comments={filteredComments}
          selectedComment={selectedComment}
          onSelectComment={handleSelectComment}
          channelFilter={channelId}
        />
        <CommentView
          comment={selectedComment}
          onSendReply={handleSendReply}
          onAIResponse={handleAIResponse}
        />
      </div>
    </div>
  );
}