import { Search } from "lucide-react";
import { Input } from "./ui/input";
import { Badge } from "./ui/badge";
import { ScrollArea } from "./ui/scroll-area";
import { Message, ChannelType } from "../types";
import { useState } from "react";

interface MessageListProps {
  messages: Message[];
  selectedMessage: Message | null;
  onSelectMessage: (message: Message) => void;
  channelFilter?: string;
}

export function MessageList({
  messages,
  selectedMessage,
  onSelectMessage,
  channelFilter,
}: MessageListProps) {
  const [searchQuery, setSearchQuery] = useState("");

  const filteredMessages = messages.filter((msg) =>
    msg.customerName.toLowerCase().includes(searchQuery.toLowerCase()) ||
    msg.lastMessage.toLowerCase().includes(searchQuery.toLowerCase())
  );

  const unreadCount = filteredMessages.reduce((sum, msg) => sum + msg.unread, 0);

  const getChannelColor = (channelType: ChannelType) => {
    const colors = {
      'trendyol-product': '#F27A00',
      'trendyol-order': '#F27A00',
      'instagram-message': '#E4405F',
      'facebook-message': '#1877F2',
      'whatsapp': '#25D366',
      'email': '#EA4335',
      'ticimax': '#00A8E8',
      'hepsiburada-product': '#FF6000',
      'hepsiburada-order': '#FF6000',
    };
    return colors[channelType];
  };

  return (
    <div className="w-80 border-r border-gray-200 bg-white flex flex-col">
      <div className="p-4 border-b border-gray-200">
        <div className="flex items-center justify-between mb-4">
          <h3 className="font-semibold text-gray-900">
            {channelFilter ? `${channelFilter.charAt(0).toUpperCase() + channelFilter.slice(1)} Mesajları` : 'Tüm Mesajlar'}
          </h3>
          <Badge variant="secondary">{filteredMessages.length}</Badge>
        </div>
        <div className="relative">
          <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
          <Input
            placeholder="Ara..."
            className="pl-10"
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
          />
        </div>
      </div>

      <ScrollArea className="flex-1">
        <div className="p-2">
          {filteredMessages.map((message) => (
            <button
              key={message.id}
              onClick={() => onSelectMessage(message)}
              className={`w-full text-left p-3 rounded-lg mb-2 transition-colors ${
                selectedMessage?.id === message.id
                  ? 'bg-blue-50 border-2 border-blue-200'
                  : 'hover:bg-gray-50 border-2 border-transparent'
              }`}
            >
              <div className="flex items-center justify-between mb-2">
                <p className="font-medium text-gray-900 text-sm">
                  {message.customerName}
                </p>
                <span className="text-xs text-gray-500">
                  {formatTime(message.timestamp)}
                </span>
              </div>
              <div className="flex items-center gap-2 flex-wrap">
                <Badge
                  variant="outline"
                  style={{ 
                    borderColor: getChannelColor(message.channelType),
                    color: getChannelColor(message.channelType),
                  }}
                  className="text-xs"
                >
                  {message.channelName}
                </Badge>
                <Badge variant="secondary" className="text-xs">
                  {message.channelCategory}
                </Badge>
                {message.unread === 0 && (
                  <Badge variant="secondary" className="bg-green-100 text-green-700 text-xs">
                    ✓ Cevaplandı
                  </Badge>
                )}
              </div>
            </button>
          ))}
        </div>
      </ScrollArea>
    </div>
  );
}

function formatTime(date: Date): string {
  const now = new Date();
  const diff = now.getTime() - date.getTime();
  const minutes = Math.floor(diff / 60000);
  const hours = Math.floor(diff / 3600000);

  if (minutes < 1) {
    return 'Şimdi';
  } else if (minutes < 60) {
    return `${minutes} dk`;
  } else if (hours < 24) {
    return `${hours} saat`;
  } else {
    return date.toLocaleDateString('tr-TR', { day: 'numeric', month: 'short' });
  }
}