import { useState } from "react";
import { mockUsers, rolePermissions } from "../data/mockData";
import { User, UserRole, UserStatus, PagePermissions } from "../types";
import { Button } from "./ui/button";
import { Input } from "./ui/input";
import { Badge } from "./ui/badge";
import { 
  Table, 
  TableBody, 
  TableCell, 
  TableHead, 
  TableHeader, 
  TableRow 
} from "./ui/table";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "./ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "./ui/select";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "./ui/alert-dialog";
import { Label } from "./ui/label";
import { Switch } from "./ui/switch";
import { 
  UserPlus, 
  Pencil, 
  Trash2, 
  Search, 
  Shield, 
  ShieldCheck, 
  User as UserIcon,
  Clock,
  Calendar,
  Eye,
  EyeOff
} from "lucide-react";
import { toast } from "sonner";

export function Users() {
  const [users, setUsers] = useState<User[]>(mockUsers);
  const [searchTerm, setSearchTerm] = useState("");
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false);
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false);
  const [isDeleteDialogOpen, setIsDeleteDialogOpen] = useState(false);
  const [selectedUser, setSelectedUser] = useState<User | null>(null);
  const [showPassword, setShowPassword] = useState(false);
  const [formData, setFormData] = useState({
    name: "",
    email: "",
    password: "",
    role: "agent" as UserRole,
    status: "active" as UserStatus,
  });

  // Arama filtresi
  const filteredUsers = users.filter(user =>
    user.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    user.email.toLowerCase().includes(searchTerm.toLowerCase())
  );

  // Rol badge renkleri
  const getRoleBadge = (role: UserRole) => {
    switch (role) {
      case "admin":
        return { icon: <Shield className="h-3 w-3 mr-1" />, variant: "destructive" as const, label: "Admin" };
      case "moderator":
        return { icon: <ShieldCheck className="h-3 w-3 mr-1" />, variant: "default" as const, label: "Moderatör" };
      case "agent":
        return { icon: <UserIcon className="h-3 w-3 mr-1" />, variant: "secondary" as const, label: "Agent" };
    }
  };

  // Durum badge renkleri
  const getStatusBadge = (status: UserStatus) => {
    return status === "active" 
      ? { variant: "outline" as const, label: "Aktif", className: "bg-green-50 text-green-700 border-green-200" }
      : { variant: "outline" as const, label: "Pasif", className: "bg-gray-100 text-gray-600 border-gray-200" };
  };

  // Kullanıcı ekleme
  const handleAddUser = () => {
    const newUser: User = {
      id: `u${users.length + 1}`,
      name: formData.name,
      email: formData.email,
      role: formData.role,
      status: formData.status,
      permissions: rolePermissions[formData.role],
      createdAt: new Date(),
    };
    setUsers([...users, newUser]);
    setIsAddDialogOpen(false);
    resetForm();
    toast.success("Kullanıcı başarıyla eklendi");
  };

  // Kullanıcı düzenleme
  const handleEditUser = () => {
    if (!selectedUser) return;
    
    const updatedUsers = users.map(user =>
      user.id === selectedUser.id
        ? {
            ...user,
            name: formData.name,
            email: formData.email,
            role: formData.role,
            status: formData.status,
            permissions: rolePermissions[formData.role],
          }
        : user
    );
    setUsers(updatedUsers);
    setIsEditDialogOpen(false);
    setSelectedUser(null);
    resetForm();
    toast.success("Kullanıcı başarıyla güncellendi");
  };

  // Kullanıcı silme
  const handleDeleteUser = () => {
    if (!selectedUser) return;
    
    setUsers(users.filter(user => user.id !== selectedUser.id));
    setIsDeleteDialogOpen(false);
    setSelectedUser(null);
    toast.success("Kullanıcı başarıyla silindi");
  };

  // Form sıfırlama
  const resetForm = () => {
    setFormData({
      name: "",
      email: "",
      password: "",
      role: "agent",
      status: "active",
    });
  };

  // Düzenleme dialogunu aç
  const openEditDialog = (user: User) => {
    setSelectedUser(user);
    setFormData({
      name: user.name,
      email: user.email,
      password: "", // Şifre alanı boş başlasın
      role: user.role,
      status: user.status,
    });
    setIsEditDialogOpen(true);
  };

  // Silme dialogunu aç
  const openDeleteDialog = (user: User) => {
    setSelectedUser(user);
    setIsDeleteDialogOpen(true);
  };

  // Tarih formatlama
  const formatDate = (date?: Date) => {
    if (!date) return "-";
    return new Intl.DateTimeFormat('tr-TR', {
      day: '2-digit',
      month: '2-digit',
      year: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    }).format(date);
  };

  // Sayfa adları Türkçe
  const pageLabels = {
    dashboard: "Dashboard",
    messages: "Mesajlar",
    comments: "Yorumlar",
    users: "Kullanıcılar",
    settings: "Ayarlar",
  };

  // İzin türü adları
  const actionLabels = {
    canView: "Görüntüleme",
    canCreate: "Oluşturma",
    canEdit: "Düzenleme",
    canDelete: "Silme",
  };

  // İzin listesi
  const permissionLabels = {
    canManageUsers: "Kullanıcı Yönetimi",
    canManageSettings: "Ayar Yönetimi",
    canViewDashboard: "Dashboard Görüntüleme",
    canViewMessages: "Mesaj Görüntüleme",
    canReplyMessages: "Mesaj Yanıtlama",
    canDeleteMessages: "Mesaj Silme",
    canViewComments: "Yorum Görüntüleme",
    canReplyComments: "Yorum Yanıtlama",
    canUseAI: "AI Kullanımı",
  };

  return (
    <div className="flex flex-col h-full bg-gray-50">
      {/* Header */}
      <div className="border-b border-gray-200 bg-white px-8 py-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-2xl font-semibold text-gray-900">Kullanıcı Yönetimi</h1>
            <p className="text-sm text-gray-500 mt-1">
              {users.length} kullanıcı • {users.filter(u => u.status === 'active').length} aktif
            </p>
          </div>
          <Button onClick={() => setIsAddDialogOpen(true)}>
            <UserPlus className="h-4 w-4 mr-2" />
            Yeni Kullanıcı
          </Button>
        </div>

        {/* Arama */}
        <div className="mt-6 relative">
          <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
          <Input
            placeholder="Kullanıcı ara (isim, e-posta)..."
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
            className="pl-10"
          />
        </div>
      </div>

      {/* Tablo */}
      <div className="flex-1 overflow-auto px-8 py-6">
        <div className="bg-white rounded-lg border border-gray-200">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>Kullanıcı</TableHead>
                <TableHead>Rol</TableHead>
                <TableHead>Durum</TableHead>
                <TableHead>Son Giriş</TableHead>
                <TableHead>Oluşturulma</TableHead>
                <TableHead className="text-right">İşlemler</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {filteredUsers.map((user) => {
                const roleBadge = getRoleBadge(user.role);
                const statusBadge = getStatusBadge(user.status);
                
                return (
                  <TableRow key={user.id}>
                    <TableCell>
                      <div className="flex items-center gap-3">
                        <div className="w-10 h-10 rounded-full bg-blue-100 flex items-center justify-center">
                          <span className="text-sm font-semibold text-blue-600">
                            {user.name.split(' ').map(n => n[0]).join('').slice(0, 2).toUpperCase()}
                          </span>
                        </div>
                        <div>
                          <p className="font-medium text-gray-900">{user.name}</p>
                          <p className="text-sm text-gray-500">{user.email}</p>
                        </div>
                      </div>
                    </TableCell>
                    <TableCell>
                      <Badge variant={roleBadge.variant} className="flex items-center w-fit">
                        {roleBadge.icon}
                        {roleBadge.label}
                      </Badge>
                    </TableCell>
                    <TableCell>
                      <Badge variant={statusBadge.variant} className={statusBadge.className}>
                        {statusBadge.label}
                      </Badge>
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-1 text-sm text-gray-600">
                        <Clock className="h-3 w-3" />
                        {formatDate(user.lastLogin)}
                      </div>
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-1 text-sm text-gray-600">
                        <Calendar className="h-3 w-3" />
                        {formatDate(user.createdAt)}
                      </div>
                    </TableCell>
                    <TableCell className="text-right">
                      <div className="flex items-center justify-end gap-2">
                        <Button
                          variant="ghost"
                          size="sm"
                          onClick={() => openEditDialog(user)}
                        >
                          <Pencil className="h-4 w-4" />
                        </Button>
                        <Button
                          variant="ghost"
                          size="sm"
                          onClick={() => openDeleteDialog(user)}
                          disabled={user.role === 'admin'}
                        >
                          <Trash2 className="h-4 w-4 text-red-600" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                );
              })}
            </TableBody>
          </Table>
        </div>
      </div>

      {/* Kullanıcı Ekleme Dialogu */}
      <Dialog open={isAddDialogOpen} onOpenChange={setIsAddDialogOpen}>
        <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Yeni Kullanıcı Ekle</DialogTitle>
            <DialogDescription>
              Sisteme yeni kullanıcı eklemek için aşağıdaki formu doldurun.
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-6 py-4">
            {/* Temel Bilgiler */}
            <div className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="add-name">Ad Soyad</Label>
                <Input
                  id="add-name"
                  placeholder="Örn: Ahmet Yılmaz"
                  value={formData.name}
                  onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                />
              </div>
              
              <div className="space-y-2">
                <Label htmlFor="add-email">E-posta</Label>
                <Input
                  id="add-email"
                  type="email"
                  placeholder="ornek@example.com"
                  value={formData.email}
                  onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="add-password">Şifre</Label>
                <div className="relative">
                  <Input
                    id="add-password"
                    type={showPassword ? "text" : "password"}
                    placeholder="••••••••"
                    value={formData.password}
                    onChange={(e) => setFormData({ ...formData, password: e.target.value })}
                  />
                  <Button
                    type="button"
                    variant="ghost"
                    size="sm"
                    className="absolute right-0 top-0 h-full px-3 hover:bg-transparent"
                    onClick={() => setShowPassword(!showPassword)}
                  >
                    {showPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                  </Button>
                </div>
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="add-role">Rol</Label>
                  <Select
                    value={formData.role}
                    onValueChange={(value: UserRole) => setFormData({ ...formData, role: value })}
                  >
                    <SelectTrigger id="add-role">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="agent">Agent</SelectItem>
                      <SelectItem value="moderator">Moderatör</SelectItem>
                      <SelectItem value="admin">Admin</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="add-status">Durum</Label>
                  <Select
                    value={formData.status}
                    onValueChange={(value: UserStatus) => setFormData({ ...formData, status: value })}
                  >
                    <SelectTrigger id="add-status">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="active">Aktif</SelectItem>
                      <SelectItem value="inactive">Pasif</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>
            </div>

            {/* İzinler Önizleme */}
            <div className="space-y-3">
              <Label>Sayfa Bazlı Yetkiler (Otomatik)</Label>
              <div className="bg-gray-50 border border-gray-200 rounded-lg overflow-hidden">
                <Table>
                  <TableHeader>
                    <TableRow className="bg-gray-100">
                      <TableHead className="font-semibold">Sayfa</TableHead>
                      <TableHead className="text-center font-semibold">Görüntüle</TableHead>
                      <TableHead className="text-center font-semibold">Oluştur</TableHead>
                      <TableHead className="text-center font-semibold">Düzenle</TableHead>
                      <TableHead className="text-center font-semibold">Sil</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {Object.entries(pageLabels).map(([key, label]) => {
                      const pagePerms = rolePermissions[formData.role][key as keyof typeof rolePermissions.admin];
                      if (typeof pagePerms === 'object' && 'canView' in pagePerms) {
                        return (
                          <TableRow key={key}>
                            <TableCell className="font-medium">{label}</TableCell>
                            <TableCell className="text-center">
                              <Badge variant={pagePerms.canView ? "default" : "secondary"} className="text-xs">
                                {pagePerms.canView ? "✓" : "✗"}
                              </Badge>
                            </TableCell>
                            <TableCell className="text-center">
                              <Badge variant={pagePerms.canCreate ? "default" : "secondary"} className="text-xs">
                                {pagePerms.canCreate ? "✓" : "✗"}
                              </Badge>
                            </TableCell>
                            <TableCell className="text-center">
                              <Badge variant={pagePerms.canEdit ? "default" : "secondary"} className="text-xs">
                                {pagePerms.canEdit ? "✓" : "✗"}
                              </Badge>
                            </TableCell>
                            <TableCell className="text-center">
                              <Badge variant={pagePerms.canDelete ? "default" : "secondary"} className="text-xs">
                                {pagePerms.canDelete ? "✓" : "✗"}
                              </Badge>
                            </TableCell>
                          </TableRow>
                        );
                      }
                      return null;
                    })}
                  </TableBody>
                </Table>
                <div className="px-4 py-3 bg-white border-t border-gray-200">
                  <div className="flex items-center justify-between text-sm">
                    <span className="text-gray-700 font-medium">AI Desteği</span>
                    <Badge variant={rolePermissions[formData.role].canUseAI ? "default" : "secondary"}>
                      {rolePermissions[formData.role].canUseAI ? "Etkin" : "Kapalı"}
                    </Badge>
                  </div>
                </div>
              </div>
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={() => {
              setIsAddDialogOpen(false);
              resetForm();
            }}>
              İptal
            </Button>
            <Button 
              onClick={handleAddUser}
              disabled={!formData.name || !formData.email}
            >
              Kullanıcı Ekle
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Kullanıcı Düzenleme Dialogu */}
      <Dialog open={isEditDialogOpen} onOpenChange={setIsEditDialogOpen}>
        <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Kullanıcıyı Düzenle</DialogTitle>
            <DialogDescription>
              Kullanıcı bilgilerini güncelleyin.
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-6 py-4">
            {/* Temel Bilgiler */}
            <div className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="edit-name">Ad Soyad</Label>
                <Input
                  id="edit-name"
                  placeholder="Örn: Ahmet Yılmaz"
                  value={formData.name}
                  onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                />
              </div>
              
              <div className="space-y-2">
                <Label htmlFor="edit-email">E-posta</Label>
                <Input
                  id="edit-email"
                  type="email"
                  placeholder="ornek@example.com"
                  value={formData.email}
                  onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="edit-password">Yeni Şifre (Opsiyonel)</Label>
                <div className="relative">
                  <Input
                    id="edit-password"
                    type={showPassword ? "text" : "password"}
                    placeholder="Değiştirmek için yeni şifre girin"
                    value={formData.password}
                    onChange={(e) => setFormData({ ...formData, password: e.target.value })}
                  />
                  <Button
                    type="button"
                    variant="ghost"
                    size="sm"
                    className="absolute right-0 top-0 h-full px-3 hover:bg-transparent"
                    onClick={() => setShowPassword(!showPassword)}
                  >
                    {showPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                  </Button>
                </div>
                <p className="text-xs text-gray-500">Boş bırakırsanız şifre değişmez</p>
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="edit-role">Rol</Label>
                  <Select
                    value={formData.role}
                    onValueChange={(value: UserRole) => setFormData({ ...formData, role: value })}
                  >
                    <SelectTrigger id="edit-role">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="agent">Agent</SelectItem>
                      <SelectItem value="moderator">Moderatör</SelectItem>
                      <SelectItem value="admin">Admin</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="edit-status">Durum</Label>
                  <Select
                    value={formData.status}
                    onValueChange={(value: UserStatus) => setFormData({ ...formData, status: value })}
                  >
                    <SelectTrigger id="edit-status">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="active">Aktif</SelectItem>
                      <SelectItem value="inactive">Pasif</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>
            </div>

            {/* İzinler Önizleme */}
            <div className="space-y-3">
              <Label>Sayfa Bazlı Yetkiler (Otomatik)</Label>
              <div className="bg-gray-50 border border-gray-200 rounded-lg overflow-hidden">
                <Table>
                  <TableHeader>
                    <TableRow className="bg-gray-100">
                      <TableHead className="font-semibold">Sayfa</TableHead>
                      <TableHead className="text-center font-semibold">Görüntüle</TableHead>
                      <TableHead className="text-center font-semibold">Oluştur</TableHead>
                      <TableHead className="text-center font-semibold">Düzenle</TableHead>
                      <TableHead className="text-center font-semibold">Sil</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {Object.entries(pageLabels).map(([key, label]) => {
                      const pagePerms = rolePermissions[formData.role][key as keyof typeof rolePermissions.admin];
                      if (typeof pagePerms === 'object' && 'canView' in pagePerms) {
                        return (
                          <TableRow key={key}>
                            <TableCell className="font-medium">{label}</TableCell>
                            <TableCell className="text-center">
                              <Badge variant={pagePerms.canView ? "default" : "secondary"} className="text-xs">
                                {pagePerms.canView ? "✓" : "✗"}
                              </Badge>
                            </TableCell>
                            <TableCell className="text-center">
                              <Badge variant={pagePerms.canCreate ? "default" : "secondary"} className="text-xs">
                                {pagePerms.canCreate ? "✓" : "✗"}
                              </Badge>
                            </TableCell>
                            <TableCell className="text-center">
                              <Badge variant={pagePerms.canEdit ? "default" : "secondary"} className="text-xs">
                                {pagePerms.canEdit ? "✓" : "✗"}
                              </Badge>
                            </TableCell>
                            <TableCell className="text-center">
                              <Badge variant={pagePerms.canDelete ? "default" : "secondary"} className="text-xs">
                                {pagePerms.canDelete ? "✓" : "✗"}
                              </Badge>
                            </TableCell>
                          </TableRow>
                        );
                      }
                      return null;
                    })}
                  </TableBody>
                </Table>
                <div className="px-4 py-3 bg-white border-t border-gray-200">
                  <div className="flex items-center justify-between text-sm">
                    <span className="text-gray-700 font-medium">AI Desteği</span>
                    <Badge variant={rolePermissions[formData.role].canUseAI ? "default" : "secondary"}>
                      {rolePermissions[formData.role].canUseAI ? "Etkin" : "Kapalı"}
                    </Badge>
                  </div>
                </div>
              </div>
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={() => {
              setIsEditDialogOpen(false);
              setSelectedUser(null);
              resetForm();
            }}>
              İptal
            </Button>
            <Button 
              onClick={handleEditUser}
              disabled={!formData.name || !formData.email}
            >
              Değişiklikleri Kaydet
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Kullanıcı Silme Dialogu */}
      <AlertDialog open={isDeleteDialogOpen} onOpenChange={setIsDeleteDialogOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Kullanıcıyı Sil</AlertDialogTitle>
            <AlertDialogDescription>
              {selectedUser?.name} adlı kullanıcıyı silmek istediğinizden emin misiniz? 
              Bu işlem geri alınamaz.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel onClick={() => setSelectedUser(null)}>
              İptal
            </AlertDialogCancel>
            <AlertDialogAction
              onClick={handleDeleteUser}
              className="bg-red-600 hover:bg-red-700"
            >
              Sil
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
}