import { useRef, useState } from "react";
import { Search, Loader2 } from "lucide-react";
import { Input } from "./ui/input";
import { Badge } from "./ui/badge";
import type { Message, ChannelType } from "../types";

interface VirtualizedMessageListProps {
  messages: Message[];
  selectedMessage: Message | null;
  onSelectMessage: (message: Message) => void;
  channelFilter?: string;
  onLoadMore: () => void;
  hasMore: boolean;
  isLoading: boolean;
}

export function VirtualizedMessageList({
  messages,
  selectedMessage,
  onSelectMessage,
  channelFilter,
  onLoadMore,
  hasMore,
  isLoading,
}: VirtualizedMessageListProps) {
  const [searchQuery, setSearchQuery] = useState("");
  const scrollContainerRef = useRef<HTMLDivElement>(null);

  const filteredMessages = messages.filter(
    (msg) =>
      msg.customerName.toLowerCase().includes(searchQuery.toLowerCase()) ||
      msg.lastMessage.toLowerCase().includes(searchQuery.toLowerCase())
  );

  function getChannelColor(channelType: ChannelType): string {
    const colors: Record<ChannelType, string> = {
      "trendyol-product": "#F27A00",
      "trendyol-order": "#F27A00",
      "instagram-message": "#E4405F",
      "facebook-message": "#1877F2",
      whatsapp: "#25D366",
      email: "#EA4335",
      ticimax: "#00A8E8",
      "hepsiburada-product": "#FF6000",
      "hepsiburada-order": "#FF6000",
    };
    return colors[channelType];
  }

  return (
    <div className="w-80 border-r border-gray-200 bg-white flex flex-col">
      <div className="p-4 border-b border-gray-200">
        <div className="flex items-center justify-between mb-4">
          <h3 className="font-semibold text-gray-900">
            {channelFilter
              ? `${channelFilter.charAt(0).toUpperCase() + channelFilter.slice(1)} Mesajları`
              : "Tüm Mesajlar"}
          </h3>
          <Badge variant="secondary">{filteredMessages.length}</Badge>
        </div>
        <div className="relative">
          <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
          <Input
            placeholder="Ara..."
            className="pl-10"
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
          />
        </div>
      </div>

      <div 
        ref={scrollContainerRef}
        className="flex-1 overflow-y-auto px-2"
        style={{ height: "calc(100vh - 200px)" }}
        onScroll={(e) => {
          if (!hasMore || isLoading) return;
          const target = e.currentTarget;
          const { scrollTop, scrollHeight, clientHeight } = target;
          if (scrollTop + clientHeight >= scrollHeight * 0.8) {
            onLoadMore();
          }
        }}
      >
        {filteredMessages.map((message) => (
          <button
            key={message.id}
            onClick={() => onSelectMessage(message)}
            className={`w-full text-left p-3 rounded-lg mb-2 transition-colors ${
              selectedMessage?.id === message.id
                ? "bg-blue-50 border-2 border-blue-200"
                : "hover:bg-gray-50 border-2 border-transparent"
            }`}
          >
            <div className="flex items-center justify-between mb-2">
              <p className="font-medium text-gray-900 text-sm truncate flex-1">
                {message.customerName}
              </p>
              <span className="text-xs text-gray-500 ml-2 flex-shrink-0">
                {formatTime(message.timestamp)}
              </span>
            </div>
            <p className="text-sm text-gray-600 truncate mb-2">
              {message.lastMessage}
            </p>
            <div className="flex items-center gap-2 flex-wrap">
              <Badge
                variant="outline"
                style={{
                  borderColor: getChannelColor(message.channelType),
                  color: getChannelColor(message.channelType),
                }}
                className="text-xs"
              >
                {message.channelName}
              </Badge>
              <Badge variant="secondary" className="text-xs">
                {message.channelCategory}
              </Badge>
              {message.isUrgent && (
                <Badge className="bg-red-500 hover:bg-red-600 text-white text-xs">
                  🚨 ACİL
                </Badge>
              )}
              {message.unread === 0 && (
                <Badge variant="secondary" className="bg-green-100 text-green-700 text-xs">
                  ✓ Cevaplandı
                </Badge>
              )}
            </div>
          </button>
        ))}

        {isLoading && (
          <div className="p-4 bg-white border-t border-gray-200 flex items-center justify-center gap-2">
            <Loader2 className="h-4 w-4 animate-spin text-blue-600" />
            <span className="text-sm text-gray-600">Yükleniyor...</span>
          </div>
        )}

        {!hasMore && filteredMessages.length > 0 && (
          <div className="p-3 bg-gray-50 border-t border-gray-200">
            <p className="text-xs text-center text-gray-500">Tüm mesajlar yüklendi</p>
          </div>
        )}

        {filteredMessages.length === 0 && (
          <div className="flex items-center justify-center h-64">
            <p className="text-gray-500 text-sm">Mesaj bulunamadı</p>
          </div>
        )}
      </div>
    </div>
  );
}

function formatTime(date: Date): string {
  const now = new Date();
  const diff = now.getTime() - date.getTime();
  const minutes = Math.floor(diff / 60000);
  const hours = Math.floor(diff / 3600000);

  if (minutes < 1) {
    return "Şimdi";
  } else if (minutes < 60) {
    return `${minutes} dk`;
  } else if (hours < 24) {
    return `${hours} saat`;
  } else {
    return date.toLocaleDateString("tr-TR", { day: "numeric", month: "short" });
  }
}